<?php

namespace App\Livewire;

use App\Models\Task;
use App\Models\Project;
use App\Models\User;
use Carbon\Carbon;
use Livewire\Component;
use Livewire\Attributes\Url;

class WeeklyTasks extends Component
{
    #[Url]
    public $selectedWeek = '';
    public $selectedProject = '';
    public $showAddTaskModal = false;
    public $newTask = [
        'title' => '',
        'project_id' => '',
        'user_id' => '',
        'priority' => 'Moyenne',
        'status' => 'À faire',
        'description' => '',
        'next_action' => ''
    ];

    public function mount()
    {
        if ($this->selectedWeek == '') {
            $this->selectedWeek = now()->format('Y-m-d');
        }
    }

    public function render()
    {
        // Récupérer tous les projets avec leurs tâches et utilisateurs
        $projects = Project::with(['tasks.user'])->get();
        $start_week = Carbon::parse($this->selectedWeek)->startOfWeek()->format('Y-m-d');
        $end_week = Carbon::parse($this->selectedWeek)->endOfWeek()->format('Y-m-d');
        // Grouper les projets par responsable
        // On veut regrouper les projets par responsable, mais inclure aussi les utilisateurs qui n'ont pas de tâches
        $users = User::all();
        $projectsByResponsible = collect();

        // D'abord, on groupe les projets par responsable comme avant
        $grouped = $projects->groupBy(function($project) {
            $latestTask = $project->tasks->sortByDesc('created_at')->first();
            return $latestTask ? $latestTask->user->name : 'Non assigné';
        });

        // On ajoute tous les utilisateurs, même ceux qui n'ont pas de projet/tâche
        foreach ($users as $user) {
            if (!$grouped->has($user->name)) {
                $projectsByResponsible->put($user->name, collect());
            } else {
                $projectsByResponsible->put($user->name, $grouped->get($user->name));
            }
        }

        // On ajoute aussi la clé "Non assigné" si elle existe dans le groupement initial
        if ($grouped->has('Non assigné')) {
            $projectsByResponsible->put('Non assigné', $grouped->get('Non assigné'));
        }

        // Récupérer les tâches de la semaine sélectionnée
        $query = Task::with(['project', 'user'])->whereBetween('week', [$start_week, $end_week]);

        if ($this->selectedWeek) {
            $query->where('week', $this->selectedWeek);
        }

        if ($this->selectedProject) {
            $query->where('project_id', $this->selectedProject);
        }

        $tasks = $query->orderBy('priority', 'desc')->get();
        $weeklyData = $tasks->groupBy('user.name');

        // Récupérer tous les utilisateurs pour le formulaire d'ajout
        $users = User::all();

        return view('livewire.weekly-tasks', [
            'weeklyData' => $weeklyData,
            'currentWeek' => $this->selectedWeek,
            'projectsByResponsible' => $projectsByResponsible,
            'users' => $users,
            'projects' => $projects
        ]);
    }

    public function previousWeek()
    {
        $this->selectedWeek = Carbon::parse($this->selectedWeek)->subWeek()->format('Y-m-d');
    }

    public function nextWeek()
    {
        $this->selectedWeek = Carbon::parse($this->selectedWeek)->addWeek()->format('Y-m-d');
    }

    public function showAddTaskModal()
    {
        $this->showAddTaskModal = true;
        $this->newTask = [
            'title' => '',
            'project_id' => '',
            'user_id' => '',
            'priority' => 'Moyenne',
            'status' => 'À faire',
            'description' => '',
            'next_action' => ''
        ];
    }

    public function closeAddTaskModal()
    {
        $this->showAddTaskModal = false;
    }

    public function addTask()
    {
        $this->validate([
            'newTask.title' => 'required|string|max:255',
            'newTask.project_id' => 'required|exists:projects,id',
            'newTask.user_id' => 'required|exists:users,id',
            'newTask.priority' => 'required|in:Haute,Moyenne,Basse',
            'newTask.status' => 'required|in:À faire,En cours,Terminé,Bloqué',
        ]);

        // Créer la tâche pour la semaine suivante
        $nextWeek = Carbon::parse($this->selectedWeek)->addWeek()->format('Y-m-d');
        
        Task::create([
            'title' => $this->newTask['title'],
            'project_id' => $this->newTask['project_id'],
            'user_id' => $this->newTask['user_id'],
            'week' => $nextWeek,
            'priority' => $this->newTask['priority'],
            'status' => $this->newTask['status'],
            'blockage' => $this->newTask['description'],
            'next_action' => $this->newTask['next_action'],
        ]);

        $this->closeAddTaskModal();
        $this->dispatch('task-added');
    }

    public function quickAddTask($responsibleName)
    {
        $user = User::where('name', $responsibleName)->first();
        $this->newTask['user_id'] = $user->id;
        $this->showAddTaskModal = true;
        /* dd($this->newTask); */
    }
}
