<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Illuminate\Http\Request;

class TaskController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $tasks = Task::with(['project', 'user'])->orderBy('created_at', 'desc')->paginate(10);
        return view('tasks.index', compact('tasks'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $projects = Project::all();
        $users = User::all();
        $statuses = ['À faire', 'En cours', 'Terminé', 'Bloqué'];
        $priorities = ['Haute', 'Moyenne', 'Basse'];
        
        return view('tasks.create', compact('projects', 'users', 'statuses', 'priorities'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'project_id' => 'required|exists:projects,id',
            'user_id' => 'required|exists:users,id',
            'week' => 'required|date',
            'status' => 'required|in:À faire,En cours,Terminé,Bloqué',
            'priority' => 'required|in:Haute,Moyenne,Basse',
            'blockage' => 'nullable|string',
            'next_action' => 'nullable|string',
        ]);

        Task::create($request->all());

        return redirect()->route('tasks.index')
            ->with('success', 'Tâche créée avec succès.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Task $task)
    {
        $task->load(['project', 'user']);
        return view('tasks.show', compact('task'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Task $task)
    {
        $projects = Project::all();
        $users = User::all();
        $statuses = ['À faire', 'En cours', 'Terminé', 'Bloqué'];
        $priorities = ['Haute', 'Moyenne', 'Basse'];
        
        return view('tasks.edit', compact('task', 'projects', 'users', 'statuses', 'priorities'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Task $task)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'project_id' => 'required|exists:projects,id',
            'user_id' => 'required|exists:users,id',
            'week' => 'required|date',
            'status' => 'required|in:À faire,En cours,Terminé,Bloqué',
            'priority' => 'required|in:Haute,Moyenne,Basse',
            'blockage' => 'nullable|string',
            'next_action' => 'nullable|string',
        ]);

        $task->update($request->all());

        return redirect()->route('tasks.index')
            ->with('success', 'Tâche mise à jour avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Task $task)
    {
        $task->delete();

        return redirect()->route('tasks.index')
            ->with('success', 'Tâche supprimée avec succès.');
    }
}
