<?php

namespace App\Livewire;

use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Carbon\Carbon;
use Livewire\Component;
use Livewire\Attributes\Url;

class WeeklyTasks extends Component
{
    
    public $selectedWeek;
    
    // Propriétés pour la modale
    public $showAddTaskModal = false;
    public $showAddProjectModal = false;
    public $newTask = [
        'title' => '',
        'project_id' => '',
        'user_id' => '',
        'status' => 'pending',
        'date_ajout' => '',
        'date_ok' => '',
        'description' => ''
    ];
    public $newProject = [
        'title' => '',
        'description' => ''
    ];

    public function mount()
    {
        if (!$this->selectedWeek) {
            $this->selectedWeek = Carbon::now()->format('Y-m-d');
        }
        $this->newTask['date_ajout'] = Carbon::now()->format('Y-m-d');
    }

    public function render()
    {
        $startOfWeek = Carbon::parse($this->selectedWeek)->startOfWeek()->format('Y-m-d');
        $endOfWeek = Carbon::parse($this->selectedWeek)->endOfWeek()->format('Y-m-d');
        $users = User::with(['tasks' => function ($query) use ($startOfWeek, $endOfWeek) {
            $query->where('date_ajout', '<=', $endOfWeek)
            ->where(function ($query) use ($startOfWeek, $endOfWeek) {
               $query->whereBetween('date_ok', [$startOfWeek, $endOfWeek])
               ->orWhere('date_ok', null);
            });
        }])->get();
        $projects = Project::all();
        $groupedUserTasks = [];
        foreach ($users as $user) {
            $groupedUserTasks[$user->name] = [];
            foreach ($projects as $project) {
                if ($user->tasks->where('project_id', $project->id)->count() > 0) {
                    $groupedUserTasks[$user->name][$project->title] = $user->tasks->where('project_id', $project->id)->first();
                    $groupedUserTasks[$user->name][$project->title]['tasks'] = $user->tasks->where('project_id', $project->id)->toArray();
                }
            }
            $groupedUserTasks[$user->name]['Autres'] = $user->tasks->where('project_id', null)->first();
            $groupedUserTasks[$user->name]['Autres']['tasks'] = $user->tasks->where('project_id', null)->toArray();
        }

        //dd($groupedUserTasks['Libasse DIOP']['Application de Gestion de Tâches']['tasks']);
       
        $currentWeek = $this->selectedWeek;
        return view('livewire.weekly-tasks', compact('currentWeek', 'users', 'groupedUserTasks', 'startOfWeek', 'endOfWeek', 'projects'));
    }

    public function previousWeek()
    {
        $this->selectedWeek = Carbon::parse($this->selectedWeek)->subWeek()->format('Y-m-d');
    }

    public function nextWeek()
    {
        $this->selectedWeek = Carbon::parse($this->selectedWeek)->addWeek()->format('Y-m-d');
    }

    public function openAddTaskModal($userName = null)
    {
        $this->showAddTaskModal = true;
        
        // Si un nom d'utilisateur est fourni, on peut pré-remplir le champ utilisateur
        if ($userName) {
            $user = User::where('name', $userName)->first();
            if ($user) {
                $this->newTask['user_id'] = $user->id;
                $this->newTask['date_ajout'] = date('Y-m-d');
            }
        }
    }

    public function closeAddTaskModal()
    {
        $this->showAddTaskModal = false;
        $this->resetNewTask();
    }

    public function resetNewTask()
    {
        $this->newTask = [
            'title' => '',
            'project_id' => '',
            'user_id' => '',
            'status' => 'pending',
            'date_ajout' => Carbon::now()->format('Y-m-d'),
            'date_ok' => '',
            'description' => ''
        ];
    }

    public function saveTask()
    {
        // Validation
        $this->validate([
            'newTask.title' => 'required|string|max:255',
            'newTask.project_id' => 'nullable|exists:projects,id',
            'newTask.user_id' => 'required|exists:users,id',
            'newTask.status' => 'required|in:pending,in_progress,completed',
            'newTask.date_ajout' => 'required|date',
            'newTask.description' => 'nullable|string|max:1000',
        ]);

        // Créer la nouvelle tâche
        Task::create([
            'title' => $this->newTask['title'],
            'description' => $this->newTask['description'],
            'project_id' => $this->newTask['project_id'] ?: null,
            'user_id' => $this->newTask['user_id'],
            'status' => $this->newTask['status'],
            'date_ajout' => $this->newTask['date_ajout'],
            'date_ok' => $this->newTask['date_ok'] ?: null,
        ]);

        // Fermer la modale et réinitialiser le formulaire
        $this->closeAddTaskModal();
        
        // Afficher un message de succès
        session()->flash('message', 'Tâche ajoutée avec succès !');
    }

    public function openAddProjectModal()
    {
        $this->showAddProjectModal = true;
    }

    public function closeAddProjectModal()
    {
        $this->showAddProjectModal = false;
        $this->resetNewProject();
    }

    public function resetNewProject()
    {
        $this->newProject = [
            'title' => '',
            'description' => ''
        ];
    }

    public function saveProject()
    {
        // Validation
        $this->validate([
            'newProject.title' => 'required|string|max:255|unique:projects,title',
            'newProject.description' => 'nullable|string|max:1000',
        ]);

        // Créer le nouveau projet
        $project = Project::create([
            'title' => $this->newProject['title'],
            'description' => $this->newProject['description'],
            'created_by' => auth()->user()->id,
        ]);

        // Mettre à jour le champ project_id dans le formulaire de tâche
        $this->newTask['project_id'] = $project->id;

        // Fermer la modale et réinitialiser le formulaire
        $this->closeAddProjectModal();
        
        // Afficher un message de succès
        session()->flash('message', 'Projet ajouté avec succès !');
    }

    public function changeTaskStatus($taskId, $status)
    {
        $task = Task::find($taskId);
        if ($status === 'completed') {
            $task->date_ok = Carbon::now()->format('Y-m-d');
        } else {
            $task->date_ok = null;
        }
        $task->status = $status;
        $task->save();
        session()->flash('message', 'Statut de la tâche modifié avec succès !');
    }
}
