<?php

namespace App\Livewire;

use App\Models\Project;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;

class ProjectsManagement extends Component
{
    use WithPagination;

    public $title = '';
    public $description = '';
    public $editingProject = null;
    public $showModal = false;
    public $isEditing = false;
    public $search = '';

    protected $rules = [
        'title' => 'required|min:3|max:255',
        'description' => 'nullable|max:1000',
    ];

    protected $messages = [
        'title.required' => 'Le titre est requis.',
        'title.min' => 'Le titre doit contenir au moins 3 caractères.',
        'title.max' => 'Le titre ne peut pas dépasser 255 caractères.',
        'description.max' => 'La description ne peut pas dépasser 1000 caractères.',
    ];

    public function render()
    {
        $projects = Project::query()
            /* ->where('created_by', Auth::id()) */
            ->when($this->search, function ($query) {
                $query->where('title', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%');
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('livewire.projects-management', [
            'projects' => $projects
        ]);
    }

    public function createProject()
    {
        $this->resetForm();
        $this->showModal = true;
        $this->isEditing = false;
    }

    public function editProject($projectId)
    {
        $project = Project::findOrFail($projectId);
        
        if ($project->created_by !== Auth::id()) {
            session()->flash('error', 'Vous n\'êtes pas autorisé à modifier ce projet.');
            return;
        }

        $this->editingProject = $project;
        $this->title = $project->title;
        $this->description = $project->description;
        $this->showModal = true;
        $this->isEditing = true;
    }

    public function saveProject()
    {
        $this->validate();

        if ($this->isEditing) {
            $this->editingProject->update([
                'title' => $this->title,
                'description' => $this->description,
            ]);
            session()->flash('success', 'Projet mis à jour avec succès.');
        } else {
            Project::create([
                'title' => $this->title,
                'description' => $this->description,
                'created_by' => Auth::id(),
            ]);
            session()->flash('success', 'Projet créé avec succès.');
        }

        $this->closeModal();
        $this->resetForm();
    }

    public function deleteProject($projectId)
    {
        $project = Project::findOrFail($projectId);
        
        if ($project->created_by !== Auth::id()) {
            session()->flash('error', 'Vous n\'êtes pas autorisé à supprimer ce projet.');
            return;
        }

        $project->delete();
        session()->flash('success', 'Projet supprimé avec succès.');
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    private function resetForm()
    {
        $this->title = '';
        $this->description = '';
        $this->editingProject = null;
        $this->isEditing = false;
        $this->resetValidation();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }
}
